<?php
/* --------------------------------------------------------------
 AdminErrorHandler.php 2020-04-16
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Admin\Application\ErrorHandling;

use Exception;
use Gambio\Core\Application\ValueObjects\UserPreferences;
use Gambio\Admin\Layout\Smarty\LayoutLoader;
use Gambio\Core\Language\TextManager;
use Psr\Http\Message\ResponseFactoryInterface;
use Psr\Http\Message\ResponseInterface;
use Slim\Exception\HttpException;
use Slim\Exception\HttpNotFoundException;
use Slim\Handlers\ErrorHandler;
use Slim\Interfaces\CallableResolverInterface;
use Smarty;
use Throwable;

/**
 * Class AdminErrorHandler
 * @package Gambio\Admin\Application\ErrorHandling
 */
class AdminErrorHandler extends ErrorHandler
{
    /**
     * @var LayoutLoader
     */
    private $layoutLoader;
    
    /**
     * @var Smarty
     */
    private $smarty;
    
    /**
     * @var TextManager
     */
    private $textManager;
    
    /**
     * @var UserPreferences
     */
    private $userPreferences;
    
    
    public function __construct(
        CallableResolverInterface $callableResolver,
        ResponseFactoryInterface $responseFactory,
        LayoutLoader $layoutLoader,
        Smarty $smarty,
        TextManager $textManager,
        UserPreferences $userPreferences
    ) {
        parent::__construct($callableResolver, $responseFactory);
        $this->layoutLoader    = $layoutLoader;
        $this->smarty          = $smarty;
        $this->textManager     = $textManager;
        $this->userPreferences = $userPreferences;
    }
    
    
    protected function respond(): ResponseInterface
    {
        if ($this->exception instanceof HttpNotFoundException) {
            return $this->notFoundResponse();
        }
        
        if ((!$this->exception instanceof HttpException)
            && ($this->exception instanceof Exception || $this->exception instanceof Throwable)) {
            return $this->serverErrorResponse();
        }
        
        return parent::respond();
    }
    
    
    private function serverErrorResponse(): ResponseInterface
    {
        $this->layoutLoader->load($this->smarty);
        
        $this->smarty->assign('content_navigation', []);
        $this->smarty->assign('page_title', $this->translate('server_error_page_title', 'http_errors'));
        $this->smarty->assign('title', $this->translate('server_error_title', 'http_errors'));
        
        $errorInformation = [
            [
                'title' => 'Type',
                'msg'   => get_class($this->exception),
            ],
            [
                'title' => 'Code',
                'msg'   => $this->exception->getCode(),
            ],
            [
                'title' => 'Message',
                'msg'   => $this->exception->getMessage(),
            ],
            [
                'title' => 'File',
                'msg'   => $this->exception->getFile(),
            ],
            [
                'title' => 'Line',
                'msg'   => $this->exception->getLine(),
            ]
        ];
        
        $this->smarty->assign('errorInformation', $errorInformation);
        $this->smarty->assign('trace', $this->exception->getTraceAsString());
        $this->smarty->assign('displayErrorDetails', $this->displayErrorDetails);
        
        $templateFile = __DIR__ . '/templates/500.html';
        $template     = $this->smarty->fetch('extends:' . $templateFile);
        
        $response = $this->responseFactory->createResponse(500);
        $response->getBody()->write($template);
        
        return $response;
    }
    
    
    private function notFoundResponse(): ResponseInterface
    {
        $this->layoutLoader->load($this->smarty);
        
        $this->smarty->assign('content_navigation', []);
        $this->smarty->assign('page_title', $this->translate('page_not_found_page_title', 'http_errors'));
        $this->smarty->assign('title', $this->translate('page_not_found_title', 'http_errors'));
        $this->smarty->assign('description', $this->translate('page_not_found_description', 'http_errors'));
        
        $templateFile = __DIR__ . '/templates/404.html';
        $template     = $this->smarty->fetch('extends:' . $templateFile);
        
        $response = $this->responseFactory->createResponse($this->statusCode);
        $response->getBody()->write($template);
        
        return $response;
    }
    
    
    private function translate(string $phrase, string $section): string
    {
        return $this->textManager->getPhraseText($phrase, $section, $this->userPreferences->languageId());
    }
}